<?php
require_once '../includes/config.php';
require_once '../includes/Auth.php';
require_once '../includes/SuperflixAPI.php';
require_once '../includes/Favorites.php';

$auth = new Auth();
$api = new SuperflixAPI();
$favorites = new Favorites();

$current_user = $auth->getCurrentUser();
$is_authenticated = $auth->isAuthenticated();

$id = $_GET['id'] ?? null;
$type = $_GET['type'] ?? 'movie';
$season = $_GET['season'] ?? 1;
$episode = $_GET['episode'] ?? 1;

if (!$id) {
    header('Location: index.php');
    exit;
}

// Obter informações do conteúdo
if ($type === 'series') {
    $content_response = $api->getSeriesInfo($id);
    $is_series = true;
} else {
    $content_response = $api->getMovieInfo($id);
    $is_series = false;
}

if (!$content_response || !isset($content_response['data'])) {
    header('Location: index.php');
    exit;
}

$content = $content_response['data'];
$is_favorite = $is_authenticated ? $favorites->isFavorite($current_user['id'], $id) : false;

// Obter URL de embed
if ($is_series) {
    $embed_url = $api->getEmbedUrl([
        'tmdb' => $id,
        'season' => $season,
        'episode' => $episode
    ]);
} else {
    $imdb_id = $content['imdb_external_id'] ?? '';
    $embed_url = $api->getEmbedUrl(['imdb' => $imdb_id]);
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($content['title']); ?> - Superflix</title>
    <link rel="stylesheet" href="../assets/css/style.css">
</head>
<body <?php echo $is_authenticated ? 'class="authenticated"' : ''; ?>>
    <!-- Header -->
    <header>
        <div class="container">
            <nav class="navbar">
                <a href="index.php" class="logo">
                    <span style="color: #1db954;">●</span> SUPERFLIX
                </a>
                
                <ul class="nav-links">
                    <li><a href="index.php">Início</a></li>
                    <li><a href="catalogo.php">Catálogo</a></li>
                </ul>

                <div class="nav-right">
                    <?php if ($is_authenticated): ?>
                        <a href="minha-lista.php" class="btn btn-secondary btn-sm">Minha Lista</a>
                    <?php else: ?>
                        <a href="login.php" class="btn btn-primary btn-sm">Login</a>
                    <?php endif; ?>
                </div>
            </nav>
        </div>
    </header>

    <!-- Detalhes -->
    <section class="section">
        <div class="container">
            <div class="row">
                <div class="col-3">
                    <img src="<?php echo $content['poster_path']; ?>" alt="<?php echo htmlspecialchars($content['title']); ?>" style="width: 100%; border-radius: 8px;">
                </div>
                <div class="col-9">
                    <h1><?php echo htmlspecialchars($content['title']); ?></h1>
                    
                    <div style="display: flex; gap: 1rem; margin-bottom: 1rem; flex-wrap: wrap;">
                        <span style="color: var(--primary); font-weight: 600;">⭐ <?php echo round($content['vote_average'], 1); ?>/10</span>
                        <span style="color: var(--text-secondary);">📅 <?php echo isset($content['release_date']) ? date('Y', strtotime($content['release_date'])) : (isset($content['first_air_date']) ? date('Y', strtotime($content['first_air_date'])) : 'N/A'); ?></span>
                        <?php if (isset($content['runtime'])): ?>
                            <span style="color: var(--text-secondary);">⏱️ <?php echo $content['runtime']; ?> min</span>
                        <?php endif; ?>
                    </div>

                    <?php if (isset($content['genres']) && !empty($content['genres'])): ?>
                        <div style="margin-bottom: 1rem;">
                            <?php foreach ($content['genres'] as $genre): ?>
                                <span style="display: inline-block; background-color: var(--bg-tertiary); padding: 0.25rem 0.75rem; border-radius: 15px; margin-right: 0.5rem; margin-bottom: 0.5rem; font-size: 0.9rem;">
                                    <?php echo htmlspecialchars($genre['name']); ?>
                                </span>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>

                    <p><?php echo htmlspecialchars($content['overview']); ?></p>

                    <div style="display: flex; gap: 1rem; margin-top: 1.5rem;">
                        <button class="btn btn-primary" onclick="playContent()">▶️ Assistir Agora</button>
                        <?php if ($is_authenticated): ?>
                            <button class="btn btn-secondary" id="favorite-btn" onclick="toggleFavorite()" style="background-color: <?php echo $is_favorite ? 'var(--primary)' : 'var(--bg-tertiary)'; ?>;">
                                ❤️ <?php echo $is_favorite ? 'Nos Favoritos' : 'Adicionar aos Favoritos'; ?>
                            </button>
                        <?php else: ?>
                            <a href="login.php" class="btn btn-secondary">❤️ Adicionar aos Favoritos</a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Reprodutor -->
    <section class="section" id="player-section" style="display: none;">
        <div class="container">
            <h2 class="section-title">Assistindo</h2>
            
            <div id="video-player" style="width: 100%; background-color: var(--bg-secondary); border-radius: 8px; overflow: hidden;">
                <iframe src="<?php echo $embed_url; ?>" width="100%" height="600" frameborder="0" allowfullscreen style="display: block;"></iframe>
            </div>

            <?php if ($is_series && isset($content['seasons'])): ?>
                <div style="margin-top: 2rem;">
                    <h3>Temporadas</h3>
                    <select onchange="changeSeason(this.value)" style="margin-top: 1rem;">
                        <?php foreach ($content['seasons'] as $s): ?>
                            <option value="<?php echo $s['season_number']; ?>" <?php echo $s['season_number'] == $season ? 'selected' : ''; ?>>
                                Temporada <?php echo $s['season_number']; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>

                    <div style="margin-top: 1rem;">
                        <h4>Episódios</h4>
                        <div class="row" style="margin-top: 1rem;">
                            <?php 
                            $current_season = null;
                            foreach ($content['seasons'] as $s) {
                                if ($s['season_number'] == $season) {
                                    $current_season = $s;
                                    break;
                                }
                            }
                            
                            if ($current_season && isset($current_season['episodes'])) {
                                foreach ($current_season['episodes'] as $ep) {
                                    echo '<div class="col-3" style="margin-bottom: 1rem;">';
                                    echo '<button class="btn btn-secondary" onclick="changeEpisode(' . $ep['episode_number'] . ')" style="width: 100%; text-align: left;">';
                                    echo 'Ep. ' . $ep['episode_number'] . ': ' . htmlspecialchars($ep['title']);
                                    echo '</button>';
                                    echo '</div>';
                                }
                            }
                            ?>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </section>

    <footer style="background-color: var(--bg-secondary); border-top: 1px solid var(--border-color); padding: 2rem 0; margin-top: 3rem;">
        <div class="container">
            <p class="text-center" style="color: var(--text-secondary);">© 2025 Superflix. Todos os direitos reservados.</p>
        </div>
    </footer>

    <script src="../assets/js/app.js"></script>
    <script>
        const contentId = <?php echo json_encode($id); ?>;
        const contentType = <?php echo json_encode($type); ?>;
        const isAuthenticated = <?php echo $is_authenticated ? 'true' : 'false'; ?>;

        function playContent() {
            document.getElementById('player-section').style.display = 'block';
            document.getElementById('video-player').scrollIntoView({ behavior: 'smooth' });
        }

        function toggleFavorite() {
            if (!isAuthenticated) {
                window.location.href = 'login.php';
                return;
            }

            const btn = document.getElementById('favorite-btn');
            const isFavorite = btn.style.backgroundColor === 'var(--primary)';

            if (isFavorite) {
                Utils.ajax('api/favorites/remove.php', 'POST', { content_id: contentId })
                    .then(response => {
                        if (response.success) {
                            btn.style.backgroundColor = 'var(--bg-tertiary)';
                            btn.textContent = '❤️ Adicionar aos Favoritos';
                            Utils.notify('Removido dos favoritos', 'success');
                        }
                    });
            } else {
                Utils.ajax('api/favorites/add.php', 'POST', {
                    content_id: contentId,
                    content_data: {
                        title: <?php echo json_encode($content['title']); ?>,
                        type: contentType,
                        poster: <?php echo json_encode($content['poster_path']); ?>
                    }
                }).then(response => {
                    if (response.success) {
                        btn.style.backgroundColor = 'var(--primary)';
                        btn.textContent = '❤️ Nos Favoritos';
                        Utils.notify('Adicionado aos favoritos', 'success');
                    }
                });
            }
        }

        function changeSeason(season) {
            window.location.href = 'detalhes.php?id=' + contentId + '&type=series&season=' + season + '&episode=1';
        }

        function changeEpisode(episode) {
            window.location.href = 'detalhes.php?id=' + contentId + '&type=series&season=<?php echo $season; ?>&episode=' + episode;
        }
    </script>
</body>
</html>
