<?php
require_once '../includes/config.php';
require_once '../includes/Auth.php';
require_once '../includes/SuperflixAPI.php';

$auth = new Auth();
$api = new SuperflixAPI();
$current_user = $auth->getCurrentUser();
$is_authenticated = $auth->isAuthenticated();

$type = $_GET['type'] ?? 'all';
$genre = $_GET['genre'] ?? '';
$sort = $_GET['sort'] ?? 'popular';
$search = $_GET['search'] ?? '';

// Obter conteúdo
$content = [];

if ($type === 'movie' || $type === 'all') {
    $movies_response = $api->getMovies();
    if ($movies_response['success']) {
        $content = array_merge($content, $movies_response['data']['movies'] ?? []);
    }
}

if ($type === 'series' || $type === 'all') {
    $series_response = $api->getSeries();
    if ($series_response['success']) {
        $content = array_merge($content, $series_response['data']['series'] ?? []);
    }
}

// Filtrar por gênero
if ($genre) {
    $content = array_filter($content, function($item) use ($genre) {
        if (!isset($item['genres'])) return false;
        foreach ($item['genres'] as $g) {
            if ($g['id'] == $genre) return true;
        }
        return false;
    });
}

// Ordenar
if ($sort === 'rating') {
    usort($content, function($a, $b) {
        return $b['vote_average'] - $a['vote_average'];
    });
} elseif ($sort === 'recent') {
    usort($content, function($a, $b) {
        $date_a = strtotime($a['release_date'] ?? $a['first_air_date'] ?? '2000-01-01');
        $date_b = strtotime($b['release_date'] ?? $b['first_air_date'] ?? '2000-01-01');
        return $date_b - $date_a;
    });
}

// Extrair gêneros únicos
$genres = [];
$all_content = array_merge(
    $api->getMovies()['data']['movies'] ?? [],
    $api->getSeries()['data']['series'] ?? []
);
foreach ($all_content as $item) {
    if (isset($item['genres'])) {
        foreach ($item['genres'] as $g) {
            if (!isset($genres[$g['id']])) {
                $genres[$g['id']] = $g['name'];
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Catálogo - Superflix</title>
    <link rel="stylesheet" href="../assets/css/style.css">
</head>
<body <?php echo $is_authenticated ? 'class="authenticated"' : ''; ?>>
    <!-- Header -->
    <header>
        <div class="container">
            <nav class="navbar">
                <a href="index.php" class="logo">
                    <span style="color: #1db954;">●</span> SUPERFLIX
                </a>
                
                <ul class="nav-links">
                    <li><a href="index.php">Início</a></li>
                    <li><a href="catalogo.php?type=movie" <?php echo $type === 'movie' ? 'class="active"' : ''; ?>>Filmes</a></li>
                    <li><a href="catalogo.php?type=series" <?php echo $type === 'series' ? 'class="active"' : ''; ?>>Séries</a></li>
                </ul>

                <div class="nav-right">
                    <form class="search-form" style="display: flex; gap: 0.5rem;">
                        <input type="text" class="search-box" placeholder="Pesquisar..." name="search">
                        <button type="submit" class="btn btn-primary btn-sm">Buscar</button>
                    </form>

                    <?php if ($is_authenticated): ?>
                        <div class="user-menu">
                            <a href="minha-lista.php" class="btn btn-secondary btn-sm">Minha Lista</a>
                            <img src="<?php echo $current_user['avatar']; ?>" alt="Avatar" class="user-avatar">
                        </div>
                    <?php else: ?>
                        <a href="login.php" class="btn btn-primary btn-sm">Login</a>
                    <?php endif; ?>
                </div>
            </nav>
        </div>
    </header>

    <!-- Filtros -->
    <section style="background-color: var(--bg-secondary); border-bottom: 1px solid var(--border-color); padding: 1.5rem 0;">
        <div class="container">
            <div class="row">
                <div class="col-3">
                    <label>Tipo</label>
                    <select name="type" onchange="updateFilters()">
                        <option value="all" <?php echo $type === 'all' ? 'selected' : ''; ?>>Todos</option>
                        <option value="movie" <?php echo $type === 'movie' ? 'selected' : ''; ?>>Filmes</option>
                        <option value="series" <?php echo $type === 'series' ? 'selected' : ''; ?>>Séries</option>
                    </select>
                </div>
                <div class="col-3">
                    <label>Gênero</label>
                    <select name="genre" onchange="updateFilters()">
                        <option value="">Todos os gêneros</option>
                        <?php foreach ($genres as $id => $name): ?>
                            <option value="<?php echo $id; ?>" <?php echo $genre == $id ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-3">
                    <label>Ordenar por</label>
                    <select name="sort" onchange="updateFilters()">
                        <option value="popular" <?php echo $sort === 'popular' ? 'selected' : ''; ?>>Popular</option>
                        <option value="rating" <?php echo $sort === 'rating' ? 'selected' : ''; ?>>Melhor Avaliado</option>
                        <option value="recent" <?php echo $sort === 'recent' ? 'selected' : ''; ?>>Mais Recente</option>
                    </select>
                </div>
            </div>
        </div>
    </section>

    <!-- Conteúdo -->
    <section class="section">
        <div class="container">
            <h2 class="section-title">
                <?php 
                if ($type === 'movie') echo 'Filmes';
                elseif ($type === 'series') echo 'Séries';
                else echo 'Catálogo Completo';
                ?>
            </h2>

            <?php if (!empty($content)): ?>
                <div class="content-grid">
                    <?php foreach ($content as $item): ?>
                        <div class="content-item">
                            <img src="<?php echo $item['poster_path']; ?>" alt="<?php echo $item['title']; ?>">
                            <div class="content-overlay">
                                <div class="content-title"><?php echo substr($item['title'], 0, 30); ?></div>
                                <div class="content-info">
                                    <span>⭐ <?php echo round($item['vote_average'], 1); ?>/10</span>
                                    <a href="detalhes.php?id=<?php echo $item['id']; ?>&type=<?php echo isset($item['seasons']) ? 'series' : 'movie'; ?>" class="btn btn-primary btn-sm">Ver</a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <p class="text-center">Nenhum conteúdo encontrado.</p>
            <?php endif; ?>
        </div>
    </section>

    <footer style="background-color: var(--bg-secondary); border-top: 1px solid var(--border-color); padding: 2rem 0; margin-top: 3rem;">
        <div class="container">
            <p class="text-center" style="color: var(--text-secondary);">© 2025 Superflix. Todos os direitos reservados.</p>
        </div>
    </footer>

    <script>
        function updateFilters() {
            const type = document.querySelector('[name="type"]').value;
            const genre = document.querySelector('[name="genre"]').value;
            const sort = document.querySelector('[name="sort"]').value;
            
            let url = 'catalogo.php?type=' + type;
            if (genre) url += '&genre=' + genre;
            if (sort) url += '&sort=' + sort;
            
            window.location.href = url;
        }
    </script>
</body>
</html>
