<?php
require_once '../includes/config.php';
require_once '../includes/Auth.php';

$auth = new Auth();

// Verificar se é admin
if (!$auth->isAdmin()) {
    header('Location: ../index.php');
    exit;
}

$current_user = $auth->getCurrentUser();
$all_users = $auth->getAllUsers();
$users_with_whatsapp = array_filter($all_users, function($u) { return !empty($u['whatsapp']) && $u['is_active']; });

$error = '';
$success = '';
$whatsapp_links = [];

// Processar envio de mensagem
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = $_POST['title'] ?? '';
    $message = $_POST['message'] ?? '';
    $target_type = $_POST['target_type'] ?? 'all';

    if (empty($title) || empty($message)) {
        $error = 'Título e mensagem são obrigatórios';
    } else {
        $recipients = [];
        
        if ($target_type === 'all') {
            $recipients = $users_with_whatsapp;
        } else {
            $selected_ids = $_POST['selected_users'] ?? [];
            foreach ($users_with_whatsapp as $user) {
                if (in_array($user['id'], $selected_ids)) {
                    $recipients[] = $user;
                }
            }
        }

        if (empty($recipients)) {
            $error = 'Nenhum usuário selecionado com WhatsApp válido';
        } else {
            foreach ($recipients as $user) {
                $whatsapp_links[] = [
                    'name' => $user['name'],
                    'whatsapp' => $user['whatsapp'],
                    'link' => 'https://wa.me/' . preg_replace('/\D/', '', $user['whatsapp']) . '?text=' . urlencode($message)
                ];
            }
            $success = 'Mensagens preparadas para ' . count($recipients) . ' usuário(s)';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Enviar Mensagens - Superflix</title>
    <link rel="stylesheet" href="../assets/css/style.css">
</head>
<body class="authenticated">
    <header>
        <div class="container">
            <nav class="navbar">
                <a href="../index.php" class="logo">
                    <span style="color: #1db954;">●</span> SUPERFLIX ADMIN
                </a>
                
                <ul class="nav-links">
                    <li><a href="dashboard.php">Dashboard</a></li>
                    <li><a href="usuarios.php">Usuários</a></li>
                    <li><a href="mensagens.php" class="active">Mensagens</a></li>
                </ul>

                <div class="nav-right">
                    <a href="../logout.php" class="btn btn-danger btn-sm">Sair</a>
                </div>
            </nav>
        </div>
    </header>

    <section class="section">
        <div class="container">
            <h2 class="section-title">Enviar Mensagens via WhatsApp</h2>

            <div class="row">
                <div class="col-6">
                    <div style="background-color: var(--bg-secondary); padding: 1.5rem; border-radius: 8px; border: 1px solid var(--border-color);">
                        <h3 style="margin-bottom: 1rem;">Compor Mensagem</h3>

                        <?php if ($error): ?>
                            <div class="alert alert-danger">
                                <span>⚠️</span>
                                <span><?php echo htmlspecialchars($error); ?></span>
                            </div>
                        <?php endif; ?>

                        <?php if ($success): ?>
                            <div class="alert alert-success">
                                <span>✓</span>
                                <span><?php echo htmlspecialchars($success); ?></span>
                            </div>
                        <?php endif; ?>

                        <form method="POST">
                            <div class="form-group">
                                <label for="title">Título da Mensagem</label>
                                <input type="text" id="title" name="title" placeholder="Ex: Novo Plano Premium" required>
                            </div>

                            <div class="form-group">
                                <label for="message">Mensagem</label>
                                <textarea id="message" name="message" placeholder="Digite sua mensagem..." required></textarea>
                                <small style="color: var(--text-secondary); display: block; margin-top: 0.5rem;">
                                    Caracteres: <span id="char-count">0</span>/1000
                                </small>
                            </div>

                            <div class="form-group">
                                <label>Enviar para</label>
                                <div style="margin-bottom: 1rem;">
                                    <label style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                                        <input type="radio" name="target_type" value="all" checked onchange="toggleUserSelection()">
                                        Todos os usuários (<?php echo count($users_with_whatsapp); ?>)
                                    </label>
                                    <label style="display: flex; align-items: center; gap: 0.5rem;">
                                        <input type="radio" name="target_type" value="selected" onchange="toggleUserSelection()">
                                        Usuários selecionados
                                    </label>
                                </div>
                            </div>

                            <div id="user-selection" style="display: none; margin-bottom: 1rem; max-height: 300px; overflow-y: auto; border: 1px solid var(--border-color); border-radius: 5px; padding: 1rem;">
                                <?php foreach ($users_with_whatsapp as $user): ?>
                                    <label style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                                        <input type="checkbox" name="selected_users[]" value="<?php echo $user['id']; ?>">
                                        <?php echo htmlspecialchars($user['name']); ?> (<?php echo htmlspecialchars($user['whatsapp']); ?>)
                                    </label>
                                <?php endforeach; ?>
                            </div>

                            <button type="submit" class="btn btn-primary btn-block">Preparar Mensagens</button>
                        </form>
                    </div>
                </div>

                <div class="col-6">
                    <div style="background-color: var(--bg-secondary); padding: 1.5rem; border-radius: 8px; border: 1px solid var(--border-color);">
                        <h3 style="margin-bottom: 1rem;">Informações</h3>
                        
                        <div style="background-color: var(--bg-tertiary); padding: 1rem; border-radius: 5px; margin-bottom: 1rem;">
                            <p style="color: var(--text-secondary); margin-bottom: 0.5rem;">
                                <strong>Total de usuários:</strong> <?php echo count($all_users); ?>
                            </p>
                            <p style="color: var(--text-secondary); margin-bottom: 0.5rem;">
                                <strong>Com WhatsApp válido:</strong> <?php echo count($users_with_whatsapp); ?>
                            </p>
                            <p style="color: var(--text-secondary);">
                                <strong>Ativos:</strong> <?php echo count(array_filter($all_users, function($u) { return $u['is_active']; })); ?>
                            </p>
                        </div>

                        <?php if (!empty($whatsapp_links)): ?>
                            <h4 style="margin-bottom: 1rem;">Links de Mensagem Preparados</h4>
                            <div style="max-height: 400px; overflow-y: auto;">
                                <?php foreach ($whatsapp_links as $link): ?>
                                    <div style="background-color: var(--bg-tertiary); padding: 0.75rem; border-radius: 5px; margin-bottom: 0.5rem;">
                                        <p style="color: var(--text-primary); margin-bottom: 0.25rem; font-weight: 500;">
                                            <?php echo htmlspecialchars($link['name']); ?>
                                        </p>
                                        <p style="color: var(--text-secondary); font-size: 0.9rem; margin-bottom: 0.5rem;">
                                            <?php echo htmlspecialchars($link['whatsapp']); ?>
                                        </p>
                                        <a href="<?php echo $link['link']; ?>" target="_blank" class="btn btn-primary btn-sm">
                                            Enviar via WhatsApp
                                        </a>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <div style="margin-top: 2rem;">
                <a href="dashboard.php" class="btn btn-secondary">Voltar ao Dashboard</a>
            </div>
        </div>
    </section>

    <footer style="background-color: var(--bg-secondary); border-top: 1px solid var(--border-color); padding: 2rem 0; margin-top: 3rem;">
        <div class="container">
            <p class="text-center" style="color: var(--text-secondary);">© 2025 Superflix Admin. Todos os direitos reservados.</p>
        </div>
    </footer>

    <script>
        // Contar caracteres
        document.getElementById('message').addEventListener('input', function(e) {
            document.getElementById('char-count').textContent = e.target.value.length;
        });

        // Toggle seleção de usuários
        function toggleUserSelection() {
            const selection = document.getElementById('user-selection');
            const isSelected = document.querySelector('[name="target_type"][value="selected"]').checked;
            selection.style.display = isSelected ? 'block' : 'none';
        }
    </script>
</body>
</html>
