<?php
/**
 * Classe de Integração com API Superflix
 * Gerencia requisições e cache de conteúdo
 */

class SuperflixAPI {
    private $cache_dir;
    private $cache_duration;

    public function __construct() {
        $this->cache_dir = CACHE_PATH;
        $this->cache_duration = CACHE_DURATION;
    }

    /**
     * Obter lista de filmes
     */
    public function getMovies() {
        return $this->cachedRequest('movies_list', function() {
            return $this->makeRequest('/movies/list');
        });
    }

    /**
     * Obter lista de séries
     */
    public function getSeries() {
        return $this->cachedRequest('series_list', function() {
            return $this->makeRequest('/series/list');
        });
    }

    /**
     * Buscar conteúdo
     */
    public function search($query, $type = 'movie') {
        $cache_key = 'search_' . md5($query . $type);
        return $this->cachedRequest($cache_key, function() use ($query, $type) {
            return $this->makeRequest('/search', [
                'query' => $query,
                'type' => $type
            ]);
        });
    }

    /**
     * Obter informações de filme
     */
    public function getMovieInfo($imdb_id) {
        $cache_key = 'movie_' . $imdb_id;
        return $this->cachedRequest($cache_key, function() use ($imdb_id) {
            return $this->makeRequest('/movies/info', ['imdb' => $imdb_id]);
        });
    }

    /**
     * Obter informações de série
     */
    public function getSeriesInfo($tmdb_id) {
        $cache_key = 'series_' . $tmdb_id;
        return $this->cachedRequest($cache_key, function() use ($tmdb_id) {
            return $this->makeRequest('/series/info', ['tmdb' => $tmdb_id]);
        });
    }

    /**
     * Obter filmes por gênero
     */
    public function getMoviesByGenre($genre_id) {
        $cache_key = 'movies_genre_' . $genre_id;
        return $this->cachedRequest($cache_key, function() use ($genre_id) {
            return $this->makeRequest('/movies/by_genre', ['genre_id' => $genre_id]);
        });
    }

    /**
     * Obter séries por gênero
     */
    public function getSeriesByGenre($genre_id) {
        $cache_key = 'series_genre_' . $genre_id;
        return $this->cachedRequest($cache_key, function() use ($genre_id) {
            return $this->makeRequest('/series/by_genre', ['genre_id' => $genre_id]);
        });
    }

    /**
     * Obter URL de embed
     */
    public function getEmbedUrl($params = []) {
        $url = SUPERFLIX_API_URL . '/embed?';
        $query_params = [];

        if (isset($params['imdb'])) {
            $query_params[] = 'imdb=' . urlencode($params['imdb']);
        }
        if (isset($params['tmdb'])) {
            $query_params[] = 'tmdb=' . urlencode($params['tmdb']);
        }
        if (isset($params['season'])) {
            $query_params[] = 'sea=' . urlencode($params['season']);
        }
        if (isset($params['episode'])) {
            $query_params[] = 'epi=' . urlencode($params['episode']);
        }

        $query_params[] = 'api_key=' . SUPERFLIX_API_KEY;

        return $url . implode('&', $query_params);
    }

    /**
     * Fazer requisição à API
     */
    private function makeRequest($endpoint, $params = []) {
        $url = SUPERFLIX_API_URL . $endpoint;
        
        // Adicionar chave de API aos parâmetros
        $params['api_key'] = SUPERFLIX_API_KEY;

        // Construir URL com query string
        if (!empty($params)) {
            $url .= '?' . http_build_query($params);
        }

        try {
            $context = stream_context_create([
                'http' => [
                    'method' => 'GET',
                    'timeout' => 10,
                    'header' => 'User-Agent: Superflix-Clone/1.0'
                ]
            ]);

            $response = @file_get_contents($url, false, $context);
            
            if ($response === false) {
                return ['success' => false, 'data' => []];
            }

            $data = json_decode($response, true);
            return $data ?? ['success' => false, 'data' => []];
        } catch (Exception $e) {
            error_log('API Error: ' . $e->getMessage());
            return ['success' => false, 'data' => []];
        }
    }

    /**
     * Requisição com cache
     */
    private function cachedRequest($cache_key, $callback) {
        $cache_file = $this->cache_dir . '/' . $cache_key . '.json';

        // Verificar se cache existe e é válido
        if (file_exists($cache_file)) {
            $file_age = time() - filemtime($cache_file);
            if ($file_age < $this->cache_duration) {
                $cached_data = file_get_contents($cache_file);
                return json_decode($cached_data, true);
            }
        }

        // Fazer requisição
        $data = $callback();

        // Salvar cache
        if ($data && isset($data['success']) && $data['success']) {
            file_put_contents($cache_file, json_encode($data, JSON_PRETTY_PRINT));
        }

        return $data;
    }

    /**
     * Limpar cache
     */
    public function clearCache() {
        $files = glob($this->cache_dir . '/*.json');
        foreach ($files as $file) {
            if (is_file($file)) {
                unlink($file);
            }
        }
        return ['success' => true, 'message' => 'Cache limpo'];
    }

    /**
     * Extrair gêneros únicos de conteúdo
     */
    public function extractGenres($content_list) {
        $genres = [];
        
        foreach ($content_list as $item) {
            if (isset($item['genres']) && is_array($item['genres'])) {
                foreach ($item['genres'] as $genre) {
                    if (!isset($genres[$genre['id']])) {
                        $genres[$genre['id']] = $genre['name'];
                    }
                }
            }
        }

        return $genres;
    }

    /**
     * Filtrar conteúdo por gênero
     */
    public function filterByGenre($content_list, $genre_id) {
        return array_filter($content_list, function($item) use ($genre_id) {
            if (!isset($item['genres'])) return false;
            foreach ($item['genres'] as $genre) {
                if ($genre['id'] == $genre_id) return true;
            }
            return false;
        });
    }
}
