<?php
/**
 * Classe de Gerenciamento de Favoritos
 * Armazena favoritos em arquivos JSON por usuário
 */

class Favorites {
    private $favorites_dir;

    public function __construct() {
        $this->favorites_dir = FAVORITES_PATH;
    }

    /**
     * Adicionar favorito
     */
    public function add($user_id, $content) {
        if (!isset($content['id'])) {
            return ['success' => false, 'message' => 'Conteúdo inválido'];
        }

        $favorites = $this->loadFavorites($user_id);
        
        // Verificar se já existe
        foreach ($favorites as $fav) {
            if ($fav['id'] === $content['id']) {
                return ['success' => false, 'message' => 'Já está nos favoritos'];
            }
        }

        // Adicionar novo favorito
        $favorite = [
            'id' => $content['id'],
            'title' => $content['title'] ?? 'Sem título',
            'type' => $content['type'] ?? 'movie',
            'poster_path' => $content['poster_path'] ?? '',
            'tmdb_id' => $content['tmdb_id'] ?? null,
            'imdb_id' => $content['imdb_id'] ?? null,
            'added_at' => date('Y-m-d H:i:s')
        ];

        $favorites[] = $favorite;
        $this->saveFavorites($user_id, $favorites);

        return ['success' => true, 'message' => 'Adicionado aos favoritos'];
    }

    /**
     * Remover favorito
     */
    public function remove($user_id, $content_id) {
        $favorites = $this->loadFavorites($user_id);
        
        $new_favorites = array_filter($favorites, function($fav) use ($content_id) {
            return $fav['id'] !== $content_id;
        });

        if (count($new_favorites) === count($favorites)) {
            return ['success' => false, 'message' => 'Favorito não encontrado'];
        }

        $this->saveFavorites($user_id, array_values($new_favorites));
        return ['success' => true, 'message' => 'Removido dos favoritos'];
    }

    /**
     * Obter todos os favoritos do usuário
     */
    public function getAll($user_id) {
        return $this->loadFavorites($user_id);
    }

    /**
     * Verificar se é favorito
     */
    public function isFavorite($user_id, $content_id) {
        $favorites = $this->loadFavorites($user_id);
        
        foreach ($favorites as $fav) {
            if ($fav['id'] === $content_id) {
                return true;
            }
        }

        return false;
    }

    /**
     * Contar favoritos
     */
    public function count($user_id) {
        return count($this->loadFavorites($user_id));
    }

    /**
     * Carregar favoritos do arquivo JSON
     */
    private function loadFavorites($user_id) {
        $file = $this->getFavoritesFile($user_id);
        
        if (!file_exists($file)) {
            return [];
        }

        $content = file_get_contents($file);
        return json_decode($content, true) ?? [];
    }

    /**
     * Salvar favoritos no arquivo JSON
     */
    private function saveFavorites($user_id, $favorites) {
        $file = $this->getFavoritesFile($user_id);
        file_put_contents($file, json_encode($favorites, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }

    /**
     * Obter caminho do arquivo de favoritos
     */
    private function getFavoritesFile($user_id) {
        return $this->favorites_dir . '/' . md5($user_id) . '.json';
    }
}
