<?php
/**
 * Classe de Autenticação - Superflix Clone
 * Gerencia login, registro e sessões de usuários com armazenamento em JSON
 */

class Auth {
    private $users_file;
    private $current_user = null;

    public function __construct() {
        $this->users_file = USERS_PATH . '/users.json';
        $this->loadCurrentUser();
    }

    /**
     * Registrar novo usuário
     */
    public function register($name, $email, $whatsapp, $password) {
        // Validações
        if (!$this->validateEmail($email)) {
            return ['success' => false, 'message' => 'Email inválido'];
        }

        if (!$this->validateWhatsapp($whatsapp)) {
            return ['success' => false, 'message' => 'WhatsApp inválido (mínimo 10 dígitos)'];
        }

        if (strlen($password) < 6) {
            return ['success' => false, 'message' => 'Senha deve ter no mínimo 6 caracteres'];
        }

        $users = $this->loadUsers();

        // Verificar se email já existe
        foreach ($users as $user) {
            if ($user['email'] === $email) {
                return ['success' => false, 'message' => 'Email já cadastrado'];
            }
        }

        // Criar novo usuário
        $new_user = [
            'id' => uniqid('user_'),
            'name' => htmlspecialchars($name),
            'email' => strtolower($email),
            'whatsapp' => preg_replace('/\D/', '', $whatsapp),
            'password' => password_hash($password, PASSWORD_BCRYPT),
            'role' => 'user',
            'is_active' => true,
            'created_at' => date('Y-m-d H:i:s'),
            'last_login' => null,
            'avatar' => 'https://ui-avatars.com/api/?name=' . urlencode($name) . '&background=1db954&color=fff'
        ];

        $users[] = $new_user;
        $this->saveUsers($users);

        // Fazer login automático
        $this->login($email, $password);

        return ['success' => true, 'message' => 'Cadastro realizado com sucesso'];
    }

    /**
     * Login de usuário
     */
    public function login($email, $password) {
        $users = $this->loadUsers();

        foreach ($users as $user) {
            if ($user['email'] === strtolower($email)) {
                if (!$user['is_active']) {
                    return ['success' => false, 'message' => 'Usuário desativado'];
                }

                if (password_verify($password, $user['password'])) {
                    // Atualizar último login
                    $user['last_login'] = date('Y-m-d H:i:s');
                    $this->updateUser($user);

                    // Criar sessão
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['user_email'] = $user['email'];
                    $_SESSION['user_role'] = $user['role'];
                    $_SESSION['login_time'] = time();

                    return ['success' => true, 'message' => 'Login realizado com sucesso'];
                }
            }
        }

        return ['success' => false, 'message' => 'Email ou senha incorretos'];
    }

    /**
     * Logout do usuário
     */
    public function logout() {
        session_destroy();
        return ['success' => true, 'message' => 'Logout realizado'];
    }

    /**
     * Verificar se usuário está autenticado
     */
    public function isAuthenticated() {
        if (!isset($_SESSION['user_id'])) {
            return false;
        }

        // Verificar timeout de sessão
        if (isset($_SESSION['login_time']) && (time() - $_SESSION['login_time']) > SESSION_TIMEOUT) {
            $this->logout();
            return false;
        }

        // Atualizar tempo de sessão
        $_SESSION['login_time'] = time();
        return true;
    }

    /**
     * Verificar se usuário é admin
     */
    public function isAdmin() {
        return $this->isAuthenticated() && isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin';
    }

    /**
     * Obter usuário atual
     */
    public function getCurrentUser() {
        if (!$this->isAuthenticated()) {
            return null;
        }

        $users = $this->loadUsers();
        foreach ($users as $user) {
            if ($user['id'] === $_SESSION['user_id']) {
                unset($user['password']); // Não retornar senha
                return $user;
            }
        }

        return null;
    }

    /**
     * Obter todos os usuários (apenas para admin)
     */
    public function getAllUsers() {
        if (!$this->isAdmin()) {
            return [];
        }

        $users = $this->loadUsers();
        foreach ($users as &$user) {
            unset($user['password']);
        }
        return $users;
    }

    /**
     * Atualizar perfil do usuário
     */
    public function updateProfile($name, $email, $whatsapp) {
        if (!$this->isAuthenticated()) {
            return ['success' => false, 'message' => 'Não autenticado'];
        }

        $users = $this->loadUsers();
        $updated = false;

        foreach ($users as &$user) {
            if ($user['id'] === $_SESSION['user_id']) {
                $user['name'] = htmlspecialchars($name);
                $user['email'] = strtolower($email);
                $user['whatsapp'] = preg_replace('/\D/', '', $whatsapp);
                $user['avatar'] = 'https://ui-avatars.com/api/?name=' . urlencode($name) . '&background=1db954&color=fff';
                $updated = true;
                break;
            }
        }

        if ($updated) {
            $this->saveUsers($users);
            $_SESSION['user_email'] = $email;
            return ['success' => true, 'message' => 'Perfil atualizado'];
        }

        return ['success' => false, 'message' => 'Erro ao atualizar perfil'];
    }

    /**
     * Atualizar status de usuário (admin)
     */
    public function updateUserStatus($user_id, $is_active) {
        if (!$this->isAdmin()) {
            return ['success' => false, 'message' => 'Acesso negado'];
        }

        $users = $this->loadUsers();
        $updated = false;

        foreach ($users as &$user) {
            if ($user['id'] === $user_id) {
                $user['is_active'] = (bool)$is_active;
                $updated = true;
                break;
            }
        }

        if ($updated) {
            $this->saveUsers($users);
            return ['success' => true, 'message' => 'Status atualizado'];
        }

        return ['success' => false, 'message' => 'Usuário não encontrado'];
    }

    /**
     * Promover usuário a admin (admin)
     */
    public function promoteToAdmin($user_id) {
        if (!$this->isAdmin()) {
            return ['success' => false, 'message' => 'Acesso negado'];
        }

        $users = $this->loadUsers();
        $updated = false;

        foreach ($users as &$user) {
            if ($user['id'] === $user_id) {
                $user['role'] = 'admin';
                $updated = true;
                break;
            }
        }

        if ($updated) {
            $this->saveUsers($users);
            return ['success' => true, 'message' => 'Usuário promovido a admin'];
        }

        return ['success' => false, 'message' => 'Usuário não encontrado'];
    }

    /**
     * Validar email
     */
    private function validateEmail($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
    }

    /**
     * Validar WhatsApp
     */
    private function validateWhatsapp($whatsapp) {
        $digits = preg_replace('/\D/', '', $whatsapp);
        return strlen($digits) >= 10 && strlen($digits) <= 15;
    }

    /**
     * Carregar usuários do arquivo JSON
     */
    private function loadUsers() {
        if (!file_exists($this->users_file)) {
            return [];
        }

        $content = file_get_contents($this->users_file);
        return json_decode($content, true) ?? [];
    }

    /**
     * Salvar usuários no arquivo JSON
     */
    private function saveUsers($users) {
        file_put_contents($this->users_file, json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }

    /**
     * Atualizar usuário
     */
    private function updateUser($updated_user) {
        $users = $this->loadUsers();
        foreach ($users as &$user) {
            if ($user['id'] === $updated_user['id']) {
                $user = $updated_user;
                break;
            }
        }
        $this->saveUsers($users);
    }

    /**
     * Carregar usuário atual
     */
    private function loadCurrentUser() {
        if ($this->isAuthenticated()) {
            $this->current_user = $this->getCurrentUser();
        }
    }
}
