/**
 * Superflix Clone - JavaScript Principal
 * Funcionalidades gerais do site
 */

// ============================================
// Utilitários
// ============================================

const Utils = {
    /**
     * Fazer requisição AJAX
     */
    ajax: function(url, method = 'GET', data = null) {
        return new Promise((resolve, reject) => {
            const xhr = new XMLHttpRequest();
            xhr.open(method, url, true);
            xhr.setRequestHeader('Content-Type', 'application/json');

            xhr.onload = function() {
                if (xhr.status >= 200 && xhr.status < 300) {
                    try {
                        resolve(JSON.parse(xhr.responseText));
                    } catch (e) {
                        resolve(xhr.responseText);
                    }
                } else {
                    reject({
                        status: xhr.status,
                        message: xhr.statusText
                    });
                }
            };

            xhr.onerror = function() {
                reject({
                    status: 0,
                    message: 'Erro de conexão'
                });
            };

            if (data) {
                xhr.send(JSON.stringify(data));
            } else {
                xhr.send();
            }
        });
    },

    /**
     * Mostrar notificação
     */
    notify: function(message, type = 'info', duration = 3000) {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <span>${message}</span>
                <button class="notification-close">&times;</button>
            </div>
        `;

        document.body.appendChild(notification);

        const closeBtn = notification.querySelector('.notification-close');
        closeBtn.addEventListener('click', () => {
            notification.remove();
        });

        setTimeout(() => {
            notification.remove();
        }, duration);
    },

    /**
     * Formatar número como moeda
     */
    formatCurrency: function(value) {
        return new Intl.NumberFormat('pt-BR', {
            style: 'currency',
            currency: 'BRL'
        }).format(value);
    },

    /**
     * Formatar data
     */
    formatDate: function(date) {
        return new Intl.DateTimeFormat('pt-BR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        }).format(new Date(date));
    },

    /**
     * Truncar texto
     */
    truncate: function(text, length = 100) {
        if (text.length > length) {
            return text.substring(0, length) + '...';
        }
        return text;
    },

    /**
     * Verificar se está autenticado
     */
    isAuthenticated: function() {
        return document.body.classList.contains('authenticated');
    }
};

// ============================================
// Modal
// ============================================

const Modal = {
    /**
     * Abrir modal
     */
    open: function(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.add('active');
            document.body.style.overflow = 'hidden';
        }
    },

    /**
     * Fechar modal
     */
    close: function(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.remove('active');
            document.body.style.overflow = 'auto';
        }
    },

    /**
     * Inicializar modais
     */
    init: function() {
        document.querySelectorAll('.modal-close').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const modal = e.target.closest('.modal');
                if (modal) {
                    modal.classList.remove('active');
                    document.body.style.overflow = 'auto';
                }
            });
        });

        document.querySelectorAll('.modal').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.classList.remove('active');
                    document.body.style.overflow = 'auto';
                }
            });
        });
    }
};

// ============================================
// Favoritos
// ============================================

const Favorites = {
    /**
     * Adicionar favorito
     */
    add: function(contentId, contentData) {
        if (!Utils.isAuthenticated()) {
            Utils.notify('Faça login para adicionar aos favoritos', 'warning');
            return;
        }

        Utils.ajax('api/favorites/add.php', 'POST', {
            content_id: contentId,
            content_data: contentData
        }).then(response => {
            if (response.success) {
                Utils.notify('Adicionado aos favoritos', 'success');
                document.querySelector(`[data-favorite="${contentId}"]`)?.classList.add('active');
            } else {
                Utils.notify(response.message || 'Erro ao adicionar', 'danger');
            }
        }).catch(error => {
            Utils.notify('Erro ao adicionar aos favoritos', 'danger');
        });
    },

    /**
     * Remover favorito
     */
    remove: function(contentId) {
        Utils.ajax('api/favorites/remove.php', 'POST', {
            content_id: contentId
        }).then(response => {
            if (response.success) {
                Utils.notify('Removido dos favoritos', 'success');
                document.querySelector(`[data-favorite="${contentId}"]`)?.classList.remove('active');
            } else {
                Utils.notify(response.message || 'Erro ao remover', 'danger');
            }
        }).catch(error => {
            Utils.notify('Erro ao remover dos favoritos', 'danger');
        });
    },

    /**
     * Verificar se é favorito
     */
    check: function(contentId) {
        Utils.ajax(`api/favorites/check.php?id=${contentId}`)
            .then(response => {
                if (response.is_favorite) {
                    document.querySelector(`[data-favorite="${contentId}"]`)?.classList.add('active');
                }
            });
    },

    /**
     * Inicializar botões de favoritos
     */
    init: function() {
        document.querySelectorAll('[data-favorite]').forEach(btn => {
            const contentId = btn.dataset.favorite;
            this.check(contentId);

            btn.addEventListener('click', (e) => {
                e.preventDefault();
                if (btn.classList.contains('active')) {
                    this.remove(contentId);
                } else {
                    this.add(contentId, {
                        title: btn.dataset.title,
                        type: btn.dataset.type,
                        poster: btn.dataset.poster
                    });
                }
            });
        });
    }
};

// ============================================
// Busca
// ============================================

const Search = {
    /**
     * Realizar busca
     */
    perform: function(query, type = 'all') {
        if (!query.trim()) {
            Utils.notify('Digite algo para buscar', 'warning');
            return;
        }

        window.location.href = `busca.php?q=${encodeURIComponent(query)}&type=${type}`;
    },

    /**
     * Inicializar busca
     */
    init: function() {
        const searchInput = document.querySelector('.search-box');
        const searchForm = document.querySelector('.search-form');

        if (searchInput) {
            searchInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    this.perform(searchInput.value);
                }
            });
        }

        if (searchForm) {
            searchForm.addEventListener('submit', (e) => {
                e.preventDefault();
                const query = searchInput.value;
                const type = document.querySelector('[name="type"]')?.value || 'all';
                this.perform(query, type);
            });
        }
    }
};

// ============================================
// Carrossel
// ============================================

const Carousel = {
    /**
     * Inicializar carrossel
     */
    init: function() {
        document.querySelectorAll('.carousel').forEach(carousel => {
            const prevBtn = carousel.querySelector('.carousel-prev');
            const nextBtn = carousel.querySelector('.carousel-next');
            const container = carousel.querySelector('.carousel-container');

            if (prevBtn && nextBtn && container) {
                prevBtn.addEventListener('click', () => {
                    container.scrollBy({ left: -300, behavior: 'smooth' });
                });

                nextBtn.addEventListener('click', () => {
                    container.scrollBy({ left: 300, behavior: 'smooth' });
                });
            }
        });
    }
};

// ============================================
// Filtros
// ============================================

const Filters = {
    /**
     * Aplicar filtros
     */
    apply: function() {
        const genre = document.querySelector('[name="genre"]')?.value;
        const type = document.querySelector('[name="type"]')?.value;
        const sort = document.querySelector('[name="sort"]')?.value;

        const params = new URLSearchParams();
        if (genre) params.append('genre', genre);
        if (type) params.append('type', type);
        if (sort) params.append('sort', sort);

        window.location.href = `catalogo.php?${params.toString()}`;
    },

    /**
     * Inicializar filtros
     */
    init: function() {
        document.querySelectorAll('[name="genre"], [name="type"], [name="sort"]').forEach(select => {
            select.addEventListener('change', () => {
                this.apply();
            });
        });
    }
};

// ============================================
// Reprodutor de Vídeo
// ============================================

const VideoPlayer = {
    /**
     * Inicializar reprodutor
     */
    init: function() {
        const playerContainer = document.getElementById('video-player');
        if (!playerContainer) return;

        const iframe = playerContainer.querySelector('iframe');
        if (iframe) {
            // Ajustar altura responsiva
            this.adjustHeight(playerContainer);
            window.addEventListener('resize', () => {
                this.adjustHeight(playerContainer);
            });
        }
    },

    /**
     * Ajustar altura do player
     */
    adjustHeight: function(container) {
        const width = container.offsetWidth;
        const height = (width / 16) * 9;
        container.style.height = height + 'px';
    },

    /**
     * Trocar servidor
     */
    switchServer: function(url) {
        const playerContainer = document.getElementById('video-player');
        if (playerContainer) {
            playerContainer.innerHTML = `<iframe src="${url}" width="100%" height="100%" frameborder="0" allowfullscreen></iframe>`;
            this.init();
        }
    }
};

// ============================================
// Inicialização
// ============================================

document.addEventListener('DOMContentLoaded', function() {
    Modal.init();
    Favorites.init();
    Search.init();
    Carousel.init();
    Filters.init();
    VideoPlayer.init();

    // Adicionar estilos de notificação
    const style = document.createElement('style');
    style.textContent = `
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 3000;
            animation: slideInRight 0.3s ease;
        }

        .notification-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            border-radius: 5px;
            background-color: var(--bg-secondary);
            border: 1px solid var(--border-color);
            color: var(--text-primary);
        }

        .notification-success .notification-content {
            border-color: var(--success);
            background-color: rgba(29, 185, 84, 0.1);
            color: var(--success);
        }

        .notification-danger .notification-content {
            border-color: var(--danger);
            background-color: rgba(231, 76, 60, 0.1);
            color: var(--danger);
        }

        .notification-warning .notification-content {
            border-color: var(--warning);
            background-color: rgba(243, 156, 18, 0.1);
            color: var(--warning);
        }

        .notification-info .notification-content {
            border-color: #3498db;
            background-color: rgba(52, 152, 219, 0.1);
            color: #3498db;
        }

        .notification-close {
            background: none;
            border: none;
            color: inherit;
            font-size: 1.5rem;
            cursor: pointer;
            padding: 0;
        }

        @keyframes slideInRight {
            from {
                transform: translateX(400px);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
    `;
    document.head.appendChild(style);
});
